﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;

public class AudioManager : MonoBehaviour
{
    //Public editor variables
    public AudioClip UISelectClip;
    public AudioClip UIBackClip;
    public AudioClip UIFocusClip;
    public AudioClip UIFocusDisabledClip;
    public AudioClip UISelectDisabledClip;
    public AudioClip UITransToAudio;
    public AudioClip UITransFromAudio;
    public AudioClip UITransBackToAudio;
    public AudioClip UITransBackFromAudio;
    public AudioClip BounceInAudio;
    public AudioClip BounceOutAudio;
    public AudioClip HintHotAudio;
    public AudioClip HintColdAudio;
    public AudioClip HiderRevealAudio;
    public AudioClip RoundVictoryAudio;
    public AudioClip VictoryAudio;
    public AudioClip ConfettiAudio;
    public AudioClip TimerTickAudio;
    public AudioClip CelebrationAudio;
    public AudioClip DefeatAudio;

    private AudioSource _MusicSource;   //The source of music

    private List<AudioSource> _SFXSources = new List<AudioSource>();    //A list of sources to play multiple SFXs at one time

    private static AudioManager _Instance;

    public static AudioManager Instance
    {
        get
        {
            if (_Instance == null)
            {
                _Instance = FindObjectOfType<AudioManager>();
            }

            return _Instance;
        }
    }

    private void Awake()
    {
        //Setup the music source
        _MusicSource = gameObject.AddComponent<AudioSource>();
        _MusicSource.loop = true;
        _MusicSource.volume = Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.MusicVolume]);
    }

    /// <summary>
    /// Plays the audio clip as a music file, on loop and at the set music volume
    /// </summary>
    /// <param name="clip">The clip to play as music</param>
    public void PlayMusic(AudioClip clip)
    {
        _MusicSource.clip = clip;
        _MusicSource.Play();
    }

    /// <summary>
    /// Stops the music source playing
    /// </summary>
    public void StopMusic()
    {
        _MusicSource.Stop();
    }

    /// <summary>
    /// Plays an audio clip as SFX at the SFX volume and can optionally be set to loop
    /// </summary>
    /// <param name="clip">The clip to play as SFX</param>
    /// <param name="shouldLoop">Whether or not to loop the clip</param>
    /// <returns>The audio source playing the SFX clip</returns>
    public AudioSource PlaySFX(AudioClip clip, bool shouldLoop = false)
    {
        AudioSource source = _SFXSources.FirstOrDefault(x => !x.isPlaying); //Get the first SFX source that isn't playing

        if (source == null || source == default(AudioSource))
        {
            //Couldn't find an SFX source that isn't playing, let's make a new one
            source = gameObject.AddComponent<AudioSource>();
            _SFXSources.Add(source);
        }

        source.volume = Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.SFXVolume]);  //Set the volume of the source to be the SFX volume setting

        if (shouldLoop)
        {
            //Loop, so set the clip and play
            source.loop = true;
            source.clip = clip;
            source.Play();
        }

        else
        {
            //Not loop, so just PlayOneShot
            source.PlayOneShot(clip);
        }

        return source;
    }

    /// <summary>
    /// Fades out the audio source over a fade time (milliseconds)
    /// </summary>
    /// <param name="source">The audio source to fade out</param>
    /// <param name="fadeTime">The duration of time (milliseconds) that the fade should take</param>
    public void FadeOutAudioSource(AudioSource source, float fadeTime)
    {
        StartCoroutine(FadeOutCoroutine(source, fadeTime));
    }

    /// <summary>
    /// Sets the volume of the music source and can be optionally set to overwrite the music volume setting
    /// </summary>
    /// <param name="volume">The new volume value to set</param>
    /// <param name="overwrite">Whether or not the existing volume setting should be overwritten</param>
    public void SetMusicVolume(float volume, bool overwrite = false)
    {
        _MusicSource.volume = volume;   //Set the volume

        if (overwrite)
        {
            //Update the setting and write to the XML
            SettingsManager.Instance.Settings[Constants.Settings.MusicVolume] = volume;
            SettingsManager.Instance.WriteSettings();
        }
    }

    /// <summary>
    /// Sets the volume of the SFX sources and can be optionally set to overwrite the SFX volume setting
    /// </summary>
    /// <param name="volume">The new volume value to set</param>
    /// <param name="overwrite">Whether or not the existing volume setting should be overwritten</param>
    public void SetSFXVolume(float volume, bool overwrite = false)
    {
        foreach (AudioSource sfxSource in _SFXSources)
        {
            //Loop through and set the volume for all SFX sources
            sfxSource.volume = volume;
        }

        if (overwrite)
        {
            //Update the setting and write to the XML
            SettingsManager.Instance.Settings[Constants.Settings.SFXVolume] = volume;
            SettingsManager.Instance.WriteSettings();
        }
    }

    /// <summary>
    /// Returns the volume of the music source (NOT the setting)
    /// </summary>
    /// <returns></returns>
    public float GetMusicVolume()
    {
        return _MusicSource.volume;
    }

    /// <summary>
    /// Returns the SFX volume setting
    /// </summary>
    /// <returns></returns>
    public float GetSFXVolume()
    {
        return Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.SFXVolume]);
    }

    private IEnumerator FadeOutCoroutine(AudioSource source, float fadeTime)
    {
        float startVolume = source.volume;

        while (source.volume > 0)
        {
            //Decrement the volume from the start to 0
            if (source.isPlaying)
            {
                source.volume -= (startVolume * (Time.deltaTime * 1000.0f)) / fadeTime;
                yield return null;
            }

            else
            {
                break;
            }
        }

        //Faded out, stop and reset the volume
        source.Stop();
        source.volume = startVolume;
    }
}
